import { l as firestoreClientGetNamedQuery, m as ensureFirestoreConfigured, Q as Query, n as Query$1, o as firestoreClientLoadBundle, p as Deferred, c as Firestore } from './database-c6191e99.js';
import 'crypto';
import '@firebase/logger';
import 'util';
import '@firebase/util';
import '@grpc/grpc-js';
import '@grpc/grpc-js/package.json';
import 'path';
import '@grpc/proto-loader';

/**
 * @license
 * Copyright 2020 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
class LoadBundleTask {
    constructor() {
        this._progressObserver = {};
        this._taskCompletionResolver = new Deferred();
        this._lastProgress = {
            taskState: 'Running',
            totalBytes: 0,
            totalDocuments: 0,
            bytesLoaded: 0,
            documentsLoaded: 0
        };
    }
    onProgress(next, error, complete) {
        this._progressObserver = {
            next,
            error,
            complete
        };
    }
    catch(onRejected) {
        return this._taskCompletionResolver.promise.catch(onRejected);
    }
    then(onFulfilled, onRejected) {
        return this._taskCompletionResolver.promise.then(onFulfilled, onRejected);
    }
    /**
     * Notifies all observers that bundle loading has completed, with a provided
     * `LoadBundleTaskProgress` object.
     */
    _completeWith(progress) {
        this._updateProgress(progress);
        if (this._progressObserver.complete) {
            this._progressObserver.complete();
        }
        this._taskCompletionResolver.resolve(progress);
    }
    /**
     * Notifies all observers that bundle loading has failed, with a provided
     * `Error` as the reason.
     */
    _failWith(error) {
        this._lastProgress.taskState = 'Error';
        if (this._progressObserver.next) {
            this._progressObserver.next(this._lastProgress);
        }
        if (this._progressObserver.error) {
            this._progressObserver.error(error);
        }
        this._taskCompletionResolver.reject(error);
    }
    /**
     * Notifies a progress update of loading a bundle.
     * @param progress - The new progress.
     */
    _updateProgress(progress) {
        this._lastProgress = progress;
        if (this._progressObserver.next) {
            this._progressObserver.next(progress);
        }
    }
}
function loadBundle(db, bundleData) {
    const resultTask = new LoadBundleTask();
    firestoreClientLoadBundle(ensureFirestoreConfigured(db._delegate), db._databaseId, bundleData, resultTask);
    return resultTask;
}
function namedQuery(db, name) {
    return firestoreClientGetNamedQuery(ensureFirestoreConfigured(db._delegate), name).then(namedQuery => {
        if (!namedQuery) {
            return null;
        }
        return new Query(db, new Query$1(db._delegate, null, namedQuery.query));
    });
}

/**
 * @license
 * Copyright 2020 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
/**
 * Prototype patches bundle loading to Firestore.
 */
function registerBundle(instance) {
    instance.prototype.loadBundle = function (data) {
        return loadBundle(this, data);
    };
    instance.prototype.namedQuery = function (queryName) {
        return namedQuery(this, queryName);
    };
}
registerBundle(Firestore);

export { registerBundle };
//# sourceMappingURL=bundle.js.map
