"use strict";
/*
 * Copyright 2020 gRPC authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.setup = exports.EdsLoadBalancer = void 0;
const load_balancer_1 = require("./load-balancer");
const load_balancing_config_1 = require("./load-balancing-config");
const load_balancer_child_handler_1 = require("./load-balancer-child-handler");
const xds_client_1 = require("./xds-client");
const channel_1 = require("./channel");
const picker_1 = require("./picker");
const constants_1 = require("./constants");
const metadata_1 = require("./metadata");
const TYPE_NAME = 'eds';
function localityToName(locality) {
    return `{region=${locality.region},zone=${locality.zone},sub_zone=${locality.sub_zone}}`;
}
/**
 * This class load balances over a cluster by making an EDS request and then
 * transforming the result into a configuration for another load balancing
 * policy.
 */
class EdsLoadBalancer {
    constructor(channelControlHelper) {
        this.channelControlHelper = channelControlHelper;
        this.xdsClient = null;
        this.edsServiceName = null;
        /**
         * Indicates whether the watcher has already been passed to this.xdsClient
         * and is getting updates.
         */
        this.isWatcherActive = false;
        this.lastestConfig = null;
        this.latestAttributes = {};
        this.latestEdsUpdate = null;
        /**
         * The priority of each locality the last time we got an update.
         */
        this.localityPriorities = new Map();
        /**
         * The name we assigned to each priority number the last time we got an
         * update.
         */
        this.priorityNames = [];
        this.nextPriorityChildNumber = 0;
        this.clusterDropStats = null;
        this.childBalancer = new load_balancer_child_handler_1.ChildLoadBalancerHandler({
            createSubchannel: (subchannelAddress, subchannelArgs) => this.channelControlHelper.createSubchannel(subchannelAddress, subchannelArgs),
            requestReresolution: () => this.channelControlHelper.requestReresolution(),
            updateState: (connectivityState, originalPicker) => {
                if (this.latestEdsUpdate === null) {
                    return;
                }
                const edsPicker = {
                    pick: (pickArgs) => {
                        var _a;
                        const dropCategory = this.checkForDrop();
                        /* If we drop the call, it ends with an UNAVAILABLE status.
                         * Otherwise, delegate picking the subchannel to the child
                         * balancer. */
                        if (dropCategory === null) {
                            return originalPicker.pick(pickArgs);
                        }
                        else {
                            (_a = this.clusterDropStats) === null || _a === void 0 ? void 0 : _a.addCallDropped(dropCategory);
                            return {
                                pickResultType: picker_1.PickResultType.DROP,
                                status: {
                                    code: constants_1.Status.UNAVAILABLE,
                                    details: `Call dropped by load balancing policy. Category: ${dropCategory}`,
                                    metadata: new metadata_1.Metadata(),
                                },
                                subchannel: null,
                                extraFilterFactory: null,
                                onCallStarted: null,
                            };
                        }
                    },
                };
                this.channelControlHelper.updateState(connectivityState, edsPicker);
            },
        });
        this.watcher = {
            onValidUpdate: (update) => {
                this.latestEdsUpdate = update;
                this.updateChild();
            },
            onResourceDoesNotExist: () => {
                var _a;
                (_a = this.xdsClient) === null || _a === void 0 ? void 0 : _a.removeEndpointWatcher(this.edsServiceName, this.watcher);
                this.isWatcherActive = false;
            },
            onTransientError: (status) => {
                if (this.latestEdsUpdate === null) {
                    channelControlHelper.updateState(channel_1.ConnectivityState.TRANSIENT_FAILURE, new picker_1.UnavailablePicker({
                        code: constants_1.Status.UNAVAILABLE,
                        details: `xDS request failed with error ${status.details}`,
                        metadata: new metadata_1.Metadata(),
                    }));
                }
            },
        };
    }
    /**
     * Check whether a single call should be dropped according to the current
     * policy, based on randomly chosen numbers. Returns the drop category if
     * the call should be dropped, and null otherwise.
     */
    checkForDrop() {
        var _a;
        if (!((_a = this.latestEdsUpdate) === null || _a === void 0 ? void 0 : _a.policy)) {
            return null;
        }
        /* The drop_overloads policy is a list of pairs of category names and
         * probabilities. For each one, if the random number is within that
         * probability range, we drop the call citing that category. Otherwise, the
         * call proceeds as usual. */
        for (const dropOverload of this.latestEdsUpdate.policy.drop_overloads) {
            if (!dropOverload.drop_percentage) {
                continue;
            }
            let randNum;
            switch (dropOverload.drop_percentage.denominator) {
                case 'HUNDRED':
                    randNum = Math.random() * 100;
                    break;
                case 'TEN_THOUSAND':
                    randNum = Math.random() * 10000;
                    break;
                case 'MILLION':
                    randNum = Math.random() * 1000000;
                    break;
                default:
                    continue;
            }
            if (randNum < dropOverload.drop_percentage.numerator) {
                return dropOverload.category;
            }
        }
        return null;
    }
    /**
     * Should be called when this balancer gets a new config and when the
     * XdsClient returns a new ClusterLoadAssignment.
     */
    updateChild() {
        var _a, _b, _c;
        if (!(this.lastestConfig && this.latestEdsUpdate)) {
            return;
        }
        /**
         * Maps each priority number to the list of localities with that priority,
         * and the list of addresses associated with each locality.
         */
        const priorityList = [];
        const newLocalityPriorities = new Map();
        /* We are given a list of localities, each of which has a priority. This
         * loop consolidates localities into buckets by priority, while also
         * simplifying the data structure to make the later steps simpler */
        for (const endpoint of this.latestEdsUpdate.endpoints) {
            let localityArray = priorityList[endpoint.priority];
            if (localityArray === undefined) {
                localityArray = [];
                priorityList[endpoint.priority] = localityArray;
            }
            const addresses = endpoint.lb_endpoints.map((lbEndpoint) => {
                /* The validator in the XdsClient class ensures that each endpoint has
                 * a socket_address with an IP address and a port_value. */
                const socketAddress = lbEndpoint.endpoint.address.socket_address;
                return {
                    host: socketAddress.address,
                    port: socketAddress.port_value,
                };
            });
            localityArray.push({
                locality: endpoint.locality,
                addresses: addresses,
                weight: (_b = (_a = endpoint.load_balancing_weight) === null || _a === void 0 ? void 0 : _a.value) !== null && _b !== void 0 ? _b : 0,
            });
            newLocalityPriorities.set(localityToName(endpoint.locality), endpoint.priority);
        }
        const newPriorityNames = [];
        const addressList = [];
        const priorityChildren = new Map();
        /* The algorithm here is as follows: for each priority we are given, from
         * high to low:
         * - If the previous mapping had any of the same localities at the same or
         *   a lower priority, use the matching name from the highest such
         *   priority, unless the new mapping has already used that name.
         * - Otherwise, construct a new name using this.nextPriorityChildNumber.
         */
        for (const [priority, localityArray] of priorityList.entries()) {
            if (localityArray === undefined) {
                continue;
            }
            /**
             * Highest (smallest number) priority value that any of the localities in
             * this locality array had a in the previous mapping.
             */
            let highestOldPriority = Infinity;
            for (const localityObj of localityArray) {
                const oldPriority = this.localityPriorities.get(localityToName(localityObj.locality));
                if (oldPriority !== undefined &&
                    oldPriority >= priority &&
                    oldPriority < highestOldPriority) {
                    highestOldPriority = oldPriority;
                }
            }
            let newPriorityName;
            if (highestOldPriority === Infinity) {
                /* No existing priority at or below the same number as the priority we
                 * are looking at had any of the localities in this priority. So, we
                 * use a new name. */
                newPriorityName = `child${this.nextPriorityChildNumber++}`;
            }
            else {
                const newName = this.priorityNames[highestOldPriority];
                if (newPriorityNames.indexOf(newName) < 0) {
                    newPriorityName = newName;
                }
                else {
                    newPriorityName = `child${this.nextPriorityChildNumber++}`;
                }
            }
            newPriorityNames[priority] = newPriorityName;
            const childTargets = new Map();
            for (const localityObj of localityArray) {
                /* Use the endpoint picking policy from the config, default to
                 * round_robin. */
                const endpointPickingPolicy = [
                    ...this.lastestConfig.eds.endpointPickingPolicy,
                    { name: 'round_robin', round_robin: {} },
                ];
                let childPolicy;
                if (this.lastestConfig.eds.lrsLoadReportingServerName) {
                    childPolicy = [
                        {
                            name: 'lrs',
                            lrs: {
                                cluster_name: this.lastestConfig.eds.cluster,
                                eds_service_name: (_c = this.lastestConfig.eds.edsServiceName) !== null && _c !== void 0 ? _c : '',
                                lrs_load_reporting_server_name: this.lastestConfig.eds
                                    .lrsLoadReportingServerName,
                                locality: localityObj.locality,
                                child_policy: endpointPickingPolicy,
                            },
                        },
                    ];
                }
                else {
                    childPolicy = endpointPickingPolicy;
                }
                childTargets.set(localityToName(localityObj.locality), {
                    weight: localityObj.weight,
                    child_policy: childPolicy,
                });
                for (const address of localityObj.addresses) {
                    addressList.push(Object.assign({ localityPath: [
                            newPriorityName,
                            localityToName(localityObj.locality),
                        ] }, address));
                }
            }
            priorityChildren.set(newPriorityName, {
                config: [
                    {
                        name: 'weighted_target',
                        weighted_target: {
                            targets: childTargets,
                        },
                    },
                ],
            });
        }
        const childConfig = {
            name: 'priority',
            priority: {
                children: priorityChildren,
                /* Contract the priority names array if it is sparse. This config only
                 * cares about the order of priorities, not their specific numbers */
                priorities: newPriorityNames.filter((value) => value !== undefined),
            },
        };
        this.childBalancer.updateAddressList(addressList, childConfig, this.latestAttributes);
        this.localityPriorities = newLocalityPriorities;
        this.priorityNames = newPriorityNames;
    }
    updateAddressList(addressList, lbConfig, attributes) {
        var _a, _b;
        if (!load_balancing_config_1.isEdsLoadBalancingConfig(lbConfig)) {
            return;
        }
        if (!(attributes.xdsClient instanceof xds_client_1.XdsClient)) {
            return;
        }
        this.lastestConfig = lbConfig;
        this.latestAttributes = attributes;
        this.xdsClient = attributes.xdsClient;
        const newEdsServiceName = (_a = lbConfig.eds.edsServiceName) !== null && _a !== void 0 ? _a : lbConfig.eds.cluster;
        /* If the name is changing, disable the old watcher before adding the new
         * one */
        if (this.isWatcherActive && this.edsServiceName !== newEdsServiceName) {
            this.xdsClient.removeEndpointWatcher(this.edsServiceName, this.watcher);
            /* Setting isWatcherActive to false here lets us have one code path for
             * calling addEndpointWatcher */
            this.isWatcherActive = false;
            /* If we have a new name, the latestEdsUpdate does not correspond to
             * the new config, so it is no longer valid */
            this.latestEdsUpdate = null;
        }
        this.edsServiceName = newEdsServiceName;
        if (!this.isWatcherActive) {
            this.xdsClient.addEndpointWatcher(this.edsServiceName, this.watcher);
            this.isWatcherActive = true;
        }
        if (lbConfig.eds.lrsLoadReportingServerName) {
            this.clusterDropStats = this.xdsClient.addClusterDropStats(lbConfig.eds.lrsLoadReportingServerName, lbConfig.eds.cluster, (_b = lbConfig.eds.edsServiceName) !== null && _b !== void 0 ? _b : '');
        }
        /* If updateAddressList is called after receiving an update and the update
         * is still valid, we want to update the child config with the information
         * in the new EdsLoadBalancingConfig. */
        this.updateChild();
    }
    exitIdle() {
        this.childBalancer.exitIdle();
    }
    resetBackoff() {
        this.childBalancer.resetBackoff();
    }
    destroy() {
        var _a;
        if (this.edsServiceName) {
            (_a = this.xdsClient) === null || _a === void 0 ? void 0 : _a.removeEndpointWatcher(this.edsServiceName, this.watcher);
        }
        this.childBalancer.destroy();
    }
    getTypeName() {
        return TYPE_NAME;
    }
}
exports.EdsLoadBalancer = EdsLoadBalancer;
function setup() {
    load_balancer_1.registerLoadBalancerType(TYPE_NAME, EdsLoadBalancer);
}
exports.setup = setup;
//# sourceMappingURL=load-balancer-eds.js.map